/*
 * Transym OCR Demonstration program
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * This program demonstrates calling TOCR version 4.0 from C#.
 *
 * Copyright (C) 2012 Transym Computer Services Ltd.
 *
 * TOCR4.0DemoCSharp Issue
 */


using System;
using System.Runtime.InteropServices;
using System.Drawing;
using System.Drawing.Imaging;
using System.Windows.Forms;
using System.Text;

namespace TOCRDemo
{
	public class main : TOCRdeclares
	{
#if X64
		private static string mSample_TIF_file = Application.StartupPath + "\\..\\..\\..\\Sample.tif";
		private static string mSample_BMP_file = Application.StartupPath + "\\..\\..\\..\\Sample.bmp";
#else
		private static string mSample_TIF_file = Application.StartupPath + "\\..\\..\\Sample.tif";
		private static string mSample_BMP_file = Application.StartupPath + "\\..\\..\\Sample.bmp";
#endif
		#region SDK Declares
		private const int DIB_PAL_COLORS = 1;
		private static uint DIB_RGB_COLORS=0;
		private static uint BI_RGB = 0;
		private const int BI_BITFIELDS = 3;
		private const int PAGE_READWRITE = 4;
		private const int FILE_MAP_WRITE = 2;
		private const int CBM_INIT = 4;
		private const int SRCCOPY = 0x00CC0020;
		
		[StructLayout(LayoutKind.Sequential, Pack=4)]
			struct RGBQUAD
		{
			public byte rgbBlue;
			public byte rgbGreen;
			public byte rgbRed;
			public byte rgbReserved;
		}

        [StructLayout(LayoutKind.Sequential, Pack = 4)]
			public struct BITMAPINFOHEADER
		{
			public uint biSize;
			public int biWidth;
			public int biHeight;
			public short biPlanes;
			public short biBitCount;
			public uint biCompression;
			public uint biSizeImage;
			public int biXPelsPerMeter;
			public int biYPelsPerMeter;
			public uint biClrUsed;
			public uint biClrImportant;
		}

        [StructLayout(LayoutKind.Sequential, Pack = 4)] 
			public struct BITMAPINFO
		{
			public BITMAPINFOHEADER bmih;
			[System.Runtime.InteropServices.MarshalAs(System.Runtime.InteropServices.UnmanagedType.ByValArray, SizeConst=2)]
			public uint[] cols;
		}

		[DllImport("kernel32.dll", CharSet=CharSet.Auto, SetLastError=true)]
		private static extern bool CloseHandle(IntPtr handle);

		[DllImport("kernel32", EntryPoint="CreateFileMappingA", CharSet=CharSet.Auto, SetLastError=true)]
		private static extern IntPtr CreateFileMappingMy(uint hFile, uint lpFileMappingAttributes, uint flProtect, uint dwMaximumSizeHigh, uint dwMaximumSizeLow, int lpName);
		
		[DllImport("kernel32", EntryPoint="MapViewOfFile", CharSet=CharSet.Auto, SetLastError=true)]
		private static extern IntPtr MapViewOfFileMy(IntPtr hFileMappingObject, uint dwDesiredAccess, uint dwFileOffsetHigh, uint dwFileOffsetLow, uint dwNumberOfBytesToMap);
		
		[DllImport("kernel32", EntryPoint="UnmapViewOfFile", CharSet=CharSet.Auto, SetLastError=true)]
		private static extern int UnmapViewOfFileMy(IntPtr lpBaseAddress);
		
		[DllImport("kernel32", EntryPoint="RtlMoveMemory", CharSet=CharSet.Auto, SetLastError=true)]
		private static extern void CopyMemory(uint lpvDest, IntPtr lpvSrc, uint cbCopy);

		[DllImport("kernel32.dll", SetLastError=true, CharSet=CharSet.Auto)]
		private static extern System.IntPtr GlobalLock(IntPtr hMem);

		[DllImport("kernel32.dll", SetLastError=true, CharSet=CharSet.Auto)]
		private static extern System.IntPtr GlobalUnlock(IntPtr hMem);
		
		[DllImport("kernel32.dll", SetLastError=true, CharSet=CharSet.Auto)]
		private static extern int GlobalFree(IntPtr hMem);
			
		[DllImport("gdi32.dll")]
		private static extern bool DeleteObject(IntPtr hObject);

		[DllImport("user32.dll")]
		private static extern IntPtr GetDC(IntPtr hwnd);

		[DllImport("gdi32.dll")]
		private static extern IntPtr CreateCompatibleDC(IntPtr hdc);

		[DllImport("user32.dll")]
		private static extern int ReleaseDC(IntPtr hwnd, IntPtr hdc);

		[DllImport("gdi32.dll")]
		private static extern int DeleteDC(IntPtr hdc);

		[DllImport("gdi32.dll")]
		private static extern IntPtr SelectObject(IntPtr hdc, IntPtr hgdiobj);

		[DllImport("gdi32.dll")]
		private static extern int BitBlt(IntPtr hdcDst, int xDst, int yDst, int w, int h, IntPtr hdcSrc, int xSrc, int ySrc, int rop);

		[DllImport("gdi32.dll")]
		private static extern IntPtr CreateDIBSection(IntPtr hdc, ref BITMAPINFO bmi, uint Usage, out IntPtr bits, IntPtr hSection, uint dwOffset); 

		#endregion

		public static void Main()
		{
            Example1(); // Demonstrates how to OCR a file
            Example2(); // Demonstrates how to OCR a multiple files
            Example3(); // Demonstrates how to OCR an image using a memory mapped file created by TOCR
            Example4(); // Demonstrates how to OCR an image using a memory mapped file created here
            Example5(); // Retrieve information on Job Slot usage
            Example6(); // Retrieve information on Job Slots
            Example7(); // Get images from a TWAIN compatible device
            Example8(); // Demonstrates TOCRSetConfig and TOCRGetConfig
		}

		// Demonstrates how to OCR a file
		private static void Example1()
		{

            int		Status;
			int		JobNo = 0;
			string	Msg = "";
			TOCRRESULTS Results = new TOCRRESULTS();
			TOCRJOBINFO2 JobInfo2 = new TOCRJOBINFO2();

			JobInfo2.ProcessOptions.DisableCharacter = new short[256];

			TOCRSetConfig(TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_MSGBOX);

			JobInfo2.InputFile = mSample_TIF_file;
			JobInfo2.JobType = TOCRJOBTYPE_TIFFFILE;

			// or
			//JobInfo2.InputFile = mSample_BMP_file;
			//JobInfo2.JobType = TOCRJOBTYPE_DIBFILE;

			Status = TOCRInitialise(ref JobNo);
			if (Status == TOCR_OK)
			{
				// or
				//if (OCRPoll(JobNo, JobInfo2))
				if (OCRWait(JobNo, JobInfo2))
				{
					if (GetResults(JobNo, ref Results))
					{
						if (FormatResults(Results, ref Msg))
						{                            
                            MessageBox.Show(Msg, "Example 1", MessageBoxButtons.OK, MessageBoxIcon.Information);
						}
					}
				}
				TOCRShutdown(JobNo);
			}
		} // Example1()

        // Demonstrates how to OCR multiple files
        private static void Example2()
        {
            int Status;
            int JobNo = 0;
            int CountDone = 0;
            TOCRRESULTS Results = new TOCRRESULTS();
            TOCRJOBINFO2 JobInfo2 = new TOCRJOBINFO2();

            TOCRSetConfig(TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_MSGBOX);

            Status = TOCRInitialise(ref JobNo);
            if (Status == TOCR_OK)
            {
                // 1st file
                JobInfo2.InputFile = mSample_TIF_file;
                JobInfo2.JobType = TOCRJOBTYPE_TIFFFILE;
                if (OCRWait(JobNo, JobInfo2))
                {
                    if (GetResults(JobNo, ref Results))
                    {
                        CountDone++;
                    }
                }

                // 2nd file
                JobInfo2.InputFile = mSample_BMP_file;
                JobInfo2.JobType = TOCRJOBTYPE_DIBFILE;
                if (OCRWait(JobNo, JobInfo2))
                {
                    if (GetResults(JobNo, ref Results))
                    {
                        CountDone++;
                    }
                }
                TOCRShutdown(JobNo);
            }
            
            MessageBox.Show(CountDone.ToString() + " of 2 jobs done" , "Example 2", MessageBoxButtons.OK, MessageBoxIcon.Information);
        } // Example2()
        
        // Demonstrates how to OCR an image using a memory mapped file created by TOCR
		private static void Example3()
		{
			int			Status;
			int			JobNo = 0;
			string		Msg = "";
			IntPtr MMFhandle = IntPtr.Zero;
			TOCRRESULTSEX Results = new TOCRRESULTSEX();
			TOCRJOBINFO2 JobInfo2 = new TOCRJOBINFO2();

			JobInfo2.ProcessOptions.DisableCharacter = new short[256];
			TOCRSetConfig(TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_MSGBOX);

			JobInfo2.JobType = TOCRJOBTYPE_MMFILEHANDLE;

			Status = TOCRInitialise(ref JobNo);
			if (Status == TOCR_OK)
			{
				Status = TOCRConvertFormat(JobNo, mSample_TIF_file, TOCRCONVERTFORMAT_TIFFFILE, ref MMFhandle, TOCRCONVERTFORMAT_MMFILEHANDLE, 0);
				if (Status == TOCR_OK)
				{
					JobInfo2.hMMF = MMFhandle;
					if (OCRWait(JobNo, JobInfo2))
					{
						if (GetResults(JobNo, ref Results))
						{
							if (FormatResults(Results, ref Msg))
							{
								MessageBox.Show(Msg, "Example 3", MessageBoxButtons.OK, MessageBoxIcon.Information);
							}
						}
					}
				}
				if (!(MMFhandle.Equals(IntPtr.Zero)))
				{
					CloseHandle(MMFhandle);
				}
				TOCRShutdown(JobNo);
			}
        } // Example3()

		// Demonstrates how to OCR an image using a memory mapped file created here
		private static void Example4()
		{
			int			Status;
			int			JobNo = 0;
			string		Msg = "";
			Bitmap		BMP;
			TOCRRESULTS Results = new TOCRRESULTS();
			TOCRJOBINFO2 JobInfo2 = new TOCRJOBINFO2();

			JobInfo2.ProcessOptions.DisableCharacter = new short[256];

			IntPtr MMFhandle = IntPtr.Zero;

			BMP = new Bitmap(mSample_BMP_file);

			MMFhandle = ConvertBitmapToMMF(BMP);
			//MMFhandle = ConvertBitmapToMMF2(BMP);

			if (!(MMFhandle.Equals(IntPtr.Zero))) 
			{
				TOCRSetConfig(TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_MSGBOX);
				JobInfo2.JobType = TOCRJOBTYPE_MMFILEHANDLE;
				Status = TOCRInitialise(ref JobNo);
				if (Status == TOCR_OK) 
				{
                    JobInfo2.hMMF = MMFhandle;
					if (OCRWait(JobNo, JobInfo2))
					{
						if (GetResults(JobNo, ref Results))
						{
							if (FormatResults(Results, ref Msg))
							{
								MessageBox.Show(Msg, "Example 4", MessageBoxButtons.OK, MessageBoxIcon.Information);
							}
						}
					}
					TOCRShutdown(JobNo);
				}
				CloseHandle(MMFhandle);
			}
        } // Example4()

		// Retrieve information on Job Slot usage
		private static void Example5()
		{
			int			Status;
			int			NumSlots;
			int[]		JobSlotInf;
			string		Msg;
			GCHandle	BytesGC;

			// uncomment to see effect on usage
			//int JobNo = 0;
			//Status = TOCRInitialise(ref JobNo);

			NumSlots = TOCRGetJobDBInfo(IntPtr.Zero);
			if (NumSlots > 0) 
			{
				JobSlotInf = new int[NumSlots];
				BytesGC = GCHandle.Alloc(JobSlotInf, GCHandleType.Pinned);
				Status = TOCRGetJobDBInfo(BytesGC.AddrOfPinnedObject());
				BytesGC.Free();
				if (Status == TOCR_OK) 
				{
					Msg = "Slot usage is" + Environment.NewLine;
					for (int SlotNo = 0; SlotNo < NumSlots; SlotNo++) 
					{
						Msg += Environment.NewLine + "Slot " + SlotNo.ToString() + " is ";
						switch (JobSlotInf[SlotNo])
						{
							case TOCRJOBSLOT_FREE:
								Msg +=  "free";
								break;
							case TOCRJOBSLOT_OWNEDBYYOU:
								Msg +=  "owned by you";
								break;
							case TOCRJOBSLOT_BLOCKEDBYYOU:
								Msg +=  "blocked by you";
								break;
							case TOCRJOBSLOT_OWNEDBYOTHER:
								Msg += "owned by another process";
								break;
							case TOCRJOBSLOT_BLOCKEDBYOTHER:
								Msg +=  "blocked by another process";
								break;
						}
					}
					MessageBox.Show(Msg, "Example 5", MessageBoxButtons.OK, MessageBoxIcon.Information);
				}
			} 
			else 
				MessageBox.Show("No Job Slots", "Example 5", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);

        } // Example5()

		// Retrieve information on Job Slots
		private static void Example6()
		{
			int			Status;
			int			NumSlots;
			string		Msg;
			int			Volume = 0;
			int			Time = 0;
			int			Remaining = 0;
			int			Features = 0;
			StringBuilder Licence = new StringBuilder(20);

			NumSlots = TOCRGetJobDBInfo(IntPtr.Zero);
			if (NumSlots > 0) 
			{
				Msg = "Slot usage is" + Environment.NewLine;
				for (int SlotNo = 0; SlotNo <= NumSlots - 1; SlotNo++) 
				{
					Msg += Environment.NewLine + "Slot " + SlotNo.ToString();
					Status = TOCRGetLicenceInfoEx(SlotNo, Licence, ref Volume, ref Time, ref Remaining, ref Features);
					if (Status == TOCR_OK)
					{
						Msg += " " + Licence;
						switch (Features)
						{
							case TOCRLICENCE_STANDARD:
								Msg += " STANDARD licence";
								break;
							case TOCRLICENCE_EURO:
								if (Licence.ToString() == "154C-43BA-9421-C925")
									Msg += " EURO TRIAL licence";
								else 
									Msg += " EURO licence";
								break;
							case TOCRLICENCE_EUROUPGRADE:
								Msg += " EURO UPGRADE licence";
								break;
							case TOCRLICENCE_V3SE:
								if (Licence.ToString() == "2E72-2B35-643A-0851")
									Msg += " V3 SE TRIAL licence";
								else 
									Msg += " V3 licence";
								break;
							case TOCRLICENCE_V3SEUPGRADE:
								Msg += " V1/2 UPGRADE to V3 SE licence";
								break;
							case TOCRLICENCE_V3PRO:
								Msg += " V3 Pro/V4 licence";
								break;
							case TOCRLICENCE_V3PROUPGRADE:
								Msg += " V1/2 UPGRADE to V3 Pro/V4 licence";
								break;
							case TOCRLICENCE_V3SEPROUPGRADE:
								Msg += " V3 SE UPGRADE to V3 Pro/V4 licence";
								break;
						}
						if (Volume != 0 | Time != 0)
						{
							Msg += " " + Remaining.ToString();
							if (Time != 0)
								Msg += " days";
							else
							{
								Msg += " A4 pages";
							}
							Msg += " remaining on licence";
						}
					}
				}
				MessageBox.Show(Msg, "Example 6", MessageBoxButtons.OK, MessageBoxIcon.Information);
			}
			else
				MessageBox.Show("No Job Slots", "Example 6", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
        } // Example6()

		// Get images from a TWAIN compatible device
		private static void Example7()
		{
			int			Status;
			int			NumImages = 0;
			int			CntImages = 0;
			IntPtr[]	hDIBs;
			GCHandle	BytesGC;
			Bitmap		BMP;

			Status = TOCRTWAINSelectDS(); // select the TWAIN device
			if (Status == TOCR_OK)
			{
				Status = TOCRTWAINShowUI(1);
				Status = TOCRTWAINAcquire(ref NumImages);
				if (Status == TOCR_OK && NumImages > 0)
				{
					hDIBs = new IntPtr[NumImages];
					BytesGC = GCHandle.Alloc(hDIBs, GCHandleType.Pinned);
					Status = TOCRTWAINGetImages(BytesGC.AddrOfPinnedObject());
					BytesGC.Free();

					for (int ImgNo = 0; ImgNo < NumImages; ImgNo++) 
					{
						// Convert the memory block to a bitmap.  If you do not require a bitmap
						// you could convert the memory block to a memory mapped file and OCR it.
						BMP = ConvertMemoryBlockToBitmap(hDIBs[ImgNo]);

						// You could combine this code with Example2 to OCR the bitmap
						if (!BMP.Equals(null))
						{
							// could save the bitmap here
							//BMP.Save("A.bmp", System.Drawing.Imaging.ImageFormat.Bmp);
							BMP.Dispose();
							BMP = null;
							CntImages ++;
						}

						// Free the global memory block
						Status = GlobalFree(hDIBs[ImgNo]);
					}
				}
			}
			MessageBox.Show(CntImages.ToString() + " images successfully received", "Example ", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
        } // Example7()

		// Demonstrates TOCRSetConfig and TOCRGetConfig
		private static void Example8()
		{
			int			JobNo = 0;
			int			Value = 0;
			StringBuilder Msg = new StringBuilder(250);

			// Override the INI file settings for all new jobs
			TOCRSetConfig(TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_MSGBOX);
			TOCRSetConfig(TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_SRV_ERRORMODE, TOCRERRORMODE_MSGBOX);

			TOCRGetConfigStr(TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_LOGFILE, Msg);
			MessageBox.Show("Default Log file name = " + Msg, "Example 8", MessageBoxButtons.OK, MessageBoxIcon.Information);

			TOCRSetConfigStr(TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_LOGFILE, "Loggederrs.lis");
			TOCRGetConfigStr(TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_LOGFILE, Msg);
			MessageBox.Show("New default Log file name = " + Msg, "Example 8", MessageBoxButtons.OK, MessageBoxIcon.Information);

			TOCRInitialise(ref JobNo);
			TOCRSetConfig(JobNo, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_NONE);

			TOCRGetConfig(JobNo, TOCRCONFIG_DLL_ERRORMODE, ref Value);
			MessageBox.Show("Job DLL error mode = " + Value.ToString(), "Example 8", MessageBoxButtons.OK, MessageBoxIcon.Information);

			TOCRGetConfig(JobNo, TOCRCONFIG_SRV_ERRORMODE, ref Value);
			MessageBox.Show("Job Service error mode = " + Value.ToString(), "Example 8", MessageBoxButtons.OK, MessageBoxIcon.Information);

			// Cause an error to be sent to Loggederrs.lis
			TOCRSetConfig(JobNo, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_LOG);
			TOCRSetConfig(JobNo, 1000, TOCRERRORMODE_LOG);

			TOCRShutdown(JobNo);

        } // Example8()

		// Wait for the engine to complete
		private static bool OCRWait(int JobNo, TOCRJOBINFO2 JobInfo2)
		{
			int		Status;
			int		JobStatus = 0;
			int		ErrorMode = 0;

			Status = TOCRDoJob2(JobNo, ref JobInfo2);
			if (Status == TOCR_OK)
			{
				Status = TOCRWaitForJob(JobNo, ref JobStatus);
			}
			
			if (Status == TOCR_OK && JobStatus == TOCRJOBSTATUS_DONE)
			{
				return true;
			}
			else
			{
				// If something has gone wrong display a message
				// (Check that the OCR engine hasn't already displayed a message)
				TOCRGetConfig(JobNo, TOCRCONFIG_DLL_ERRORMODE, ref ErrorMode);
				if (ErrorMode == TOCRERRORMODE_NONE)
				{
					StringBuilder Msg = new StringBuilder(TOCRJOBMSGLENGTH);
					TOCRGetJobStatusMsg(JobNo, Msg);
					MessageBox.Show(Msg.ToString(), "OCRWait", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
				}
				return false;
			}
		}
		
		// Wait for the engine to complete by polling
		private static bool OCRPoll(int JobNo, TOCRJOBINFO2 JobInfo2)
		{
			int			Status;
			int			JobStatus = 0;
			int			ErrorMode = 0;
			float		Progress = 0;
			int			AutoOrientation = 0;
	
			Status = TOCRDoJob2(JobNo, ref JobInfo2);
			if (Status == TOCR_OK)
			{
				do
				{
					//Status = TOCRGetJobStatus(JobNo, JobStatus)
					Status = TOCRGetJobStatusEx(JobNo, ref JobStatus, ref Progress, ref AutoOrientation);

					// Do something whilst the OCR engine runs
                    Application.DoEvents(); System.Threading.Thread.Sleep(100); Application.DoEvents();
					Console.WriteLine("Progress" + Convert.ToString(Convert.ToInt32(Progress * 100)) + "%");
				} while (Status == TOCR_OK & JobStatus == TOCRJOBSTATUS_BUSY);

			}

			if (Status == TOCR_OK & JobStatus == TOCRJOBSTATUS_DONE)
			{
				return true;
			}
			else
			{
				// If something has gone wrong display a message
				// (Check that the OCR engine hasn't already displayed a message)
				TOCRGetConfig(JobNo, TOCRCONFIG_DLL_ERRORMODE, ref ErrorMode);
				if (ErrorMode == TOCRERRORMODE_NONE)
				{
					StringBuilder Msg = new StringBuilder(TOCRJOBMSGLENGTH);
					TOCRGetJobStatusMsg(JobNo, Msg);
					MessageBox.Show(Msg.ToString(), "OCRPoll", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
				}
				return false;
			}
		}

		// OVERLOADED function to retrieve the results from the service process and load into 'Results'
		// Remember the character numbers returned refer to the Windows character set.
		private static bool GetResults(int JobNo, ref TOCRRESULTS Results)
		{
			int			ResultsInf = 0; // number of bytes needed for results
			byte[]		Bytes;
			int 		Offset;
			bool		RetStatus = false;
			GCHandle	BytesGC;
			System.IntPtr AddrOfItemBytes;

			Results.Hdr.NumItems = 0;
			if (TOCRGetJobResults(JobNo, ref ResultsInf, IntPtr.Zero) == TOCR_OK) 
			{
				if (ResultsInf > 0) 
				{
					Bytes = new Byte[ResultsInf - 1];
					// pin the Bytes array so that TOCRGetJobResults can write to it
					BytesGC = GCHandle.Alloc(Bytes, GCHandleType.Pinned);
					if (TOCRGetJobResults(JobNo, ref ResultsInf, BytesGC.AddrOfPinnedObject()) == TOCR_OK) 
					{
						Results.Hdr = ((TOCRRESULTSHEADER)(Marshal.PtrToStructure(BytesGC.AddrOfPinnedObject(), typeof(TOCRRESULTSHEADER))));
						if (Results.Hdr.NumItems > 0) 
						{
							Results.Item = new TOCRRESULTSITEM[Results.Hdr.NumItems];
							Offset = Marshal.SizeOf(typeof(TOCRRESULTSHEADER));
							for (int ItemNo = 0; ItemNo <= Results.Hdr.NumItems - 1; ItemNo++) 
							{
								AddrOfItemBytes = Marshal.UnsafeAddrOfPinnedArrayElement(Bytes, Offset);
								Results.Item[ItemNo] = ((TOCRRESULTSITEM)(Marshal.PtrToStructure(AddrOfItemBytes, typeof(TOCRRESULTSITEM))));
								Offset = Offset + Marshal.SizeOf(typeof(TOCRRESULTSITEM));
							}
						}

						RetStatus = true;

						// Double check results
						if (Results.Hdr.StructId == 0) 
						{
							if (Results.Hdr.NumItems > 0) 
							{
								if (Results.Item[0].StructId != 0) 
								{
									MessageBox.Show("Wrong results item structure Id " + Results.Item[0].StructId.ToString(), "GetResults", MessageBoxButtons.OK, MessageBoxIcon.Stop);
									Results.Hdr.NumItems = 0;
									RetStatus = false;
								}
							}
						} 
						else 
						{
							MessageBox.Show("Wrong results header structure Id " + Results.Item[0].StructId.ToString(), "GetResults", MessageBoxButtons.OK, MessageBoxIcon.Stop);
							Results.Hdr.NumItems = 0;
							RetStatus = false;
						}
					}
					BytesGC.Free();
				}
			}
			return RetStatus;
		}

		// copy of TOCRRESULTSITEMEX without the Alt[] array 
		[StructLayout(LayoutKind.Sequential, Pack=4)]
			struct TOCRRESULTSITEMEXHDR
		{
			public short StructId;
			public short OCRCha;
			public float Confidence;
			public short XPos;
			public short YPos;
			public short XDim;
			public short YDim;
		}

		// OVERLOADED function to retrieve the results from the service process and load into 'ResultsEx'
		// Remember the character numbers returned refer to the Windows character set.
		private static bool GetResults(int JobNo, ref TOCRRESULTSEX ResultsEx)
		{
			int			ResultsInf = 0;
			byte[]		Bytes;
			int			Offset;
			bool		RetStatus = false;

			TOCRRESULTSITEMEXHDR ItemHdr;
			GCHandle	BytesGC;
			System.IntPtr AddrOfItemBytes;

			ResultsEx.Hdr.NumItems = 0;
			if (TOCRGetJobResultsEx(JobNo, TOCRGETRESULTS_EXTENDED, ref ResultsInf, IntPtr.Zero) == TOCR_OK) 
			{
				if (ResultsInf > 0) 
				{
					Bytes = new Byte[ResultsInf - 1];
					// pin the Bytes array so that TOCRGetJobResultsEx can write to it
					BytesGC = GCHandle.Alloc(Bytes, GCHandleType.Pinned);
					if (TOCRGetJobResultsEx(JobNo, TOCRGETRESULTS_EXTENDED, ref ResultsInf, BytesGC.AddrOfPinnedObject()) == TOCR_OK) 
					{
						ResultsEx.Hdr = ((TOCRRESULTSHEADER)(Marshal.PtrToStructure(BytesGC.AddrOfPinnedObject(), typeof(TOCRRESULTSHEADER))));
						if (ResultsEx.Hdr.NumItems > 0) 
						{
							ResultsEx.Item = new TOCRRESULTSITEMEX[ResultsEx.Hdr.NumItems];
							Offset = Marshal.SizeOf(typeof(TOCRRESULTSHEADER));
							for (int ItemNo = 0; ItemNo <= ResultsEx.Hdr.NumItems - 1; ItemNo++) 
							{
								ResultsEx.Item[ItemNo].Alt = new TOCRRESULTSITEMEXALT[5];
								
								// Cannot Marshal TOCRRESULTSITEMEX so use copy of structure header
								// This unfortunately means a double copy of the data
								AddrOfItemBytes = Marshal.UnsafeAddrOfPinnedArrayElement(Bytes, Offset);
								ItemHdr = ((TOCRRESULTSITEMEXHDR)(Marshal.PtrToStructure(AddrOfItemBytes, typeof(TOCRRESULTSITEMEXHDR))));
								ResultsEx.Item[ItemNo].StructId	  = ItemHdr.StructId;
								ResultsEx.Item[ItemNo].OCRCha	  = ItemHdr.OCRCha;
								ResultsEx.Item[ItemNo].Confidence = ItemHdr.Confidence;
								ResultsEx.Item[ItemNo].XPos		  = ItemHdr.XPos;
								ResultsEx.Item[ItemNo].YPos		  = ItemHdr.YPos;
								ResultsEx.Item[ItemNo].XDim		  = ItemHdr.XDim;
								ResultsEx.Item[ItemNo].YDim		  = ItemHdr.YDim;
								Offset = Offset + Marshal.SizeOf(typeof(TOCRRESULTSITEMEXHDR));

								for ( int AltNo = 0; AltNo < 5; AltNo++) 
								{
									AddrOfItemBytes = Marshal.UnsafeAddrOfPinnedArrayElement(Bytes, Offset);
									ResultsEx.Item[ItemNo].Alt[AltNo] = ((TOCRRESULTSITEMEXALT)(Marshal.PtrToStructure(AddrOfItemBytes, typeof(TOCRRESULTSITEMEXALT))));
									Offset = Offset + Marshal.SizeOf(typeof(TOCRRESULTSITEMEXALT));
								}
							}
						}

						RetStatus = true;

						// Double check results
						if (ResultsEx.Hdr.StructId == 0) 
						{
							if (ResultsEx.Hdr.NumItems > 0) 
							{
								if (ResultsEx.Item[0].StructId != 0) 
								{
									MessageBox.Show("Wrong results item structure Id " + ResultsEx.Item[0].StructId.ToString(), "GetResultsEx", MessageBoxButtons.OK, MessageBoxIcon.Stop);
									ResultsEx.Hdr.NumItems = 0;
									RetStatus = false;
								}
							}
						} 
						else 
						{
							MessageBox.Show("Wrong results header structure Id " + ResultsEx.Item[0].StructId.ToString(), "GetResults", MessageBoxButtons.OK, MessageBoxIcon.Stop);
							ResultsEx.Hdr.NumItems = 0;
							RetStatus = false;
						}
					}
					BytesGC.Free();
				}
			}
			return RetStatus;
		}

		// OVERLOADED function to convert results to a string
		private static bool FormatResults(TOCRRESULTS Results, ref string Msg)
		{
			if (Results.Hdr.NumItems > 0)
			{
                for (int ItemNo = 0; ItemNo < Results.Hdr.NumItems; ItemNo++)
                {
                    if (Results.Item[ItemNo].OCRCha == 13)
                        Msg = Msg + Environment.NewLine;
                    else
                        Msg = Msg + Convert.ToChar(Results.Item[ItemNo].OCRCha);
                }

                /*Encoding UNICODE = Encoding.Unicode;
                Encoding ANSI = Encoding.GetEncoding(1252);
                byte[] ansibytes = new Byte[Results.Hdr.NumItems];
                byte[] unicodebytes = new Byte[Results.Hdr.NumItems];

                for (int ItemNo = 0; ItemNo < Results.Hdr.NumItems; ItemNo++)
				{
                    if (Results.Item[ItemNo].OCRCha == 13)
                        ansibytes[ItemNo] = (byte)'\n';
                    else
                    {
                        ansibytes[ItemNo] = (byte)Results.Item[ItemNo].OCRCha;
                    }
				}
                unicodebytes = Encoding.Convert(ANSI, UNICODE, ansibytes);
                Msg = UNICODE.GetString(unicodebytes);
                */
                return true;
			}
			else
			{
				MessageBox.Show("No results returned", "FormatResults", MessageBoxButtons.OK, MessageBoxIcon.Information);
				return false;
			}
		}

		// OVERLOADED function to convert results to a string
		private static bool FormatResults(TOCRRESULTSEX ResultsEx, ref string Msg)
		{
			Msg = "";

			if (ResultsEx.Hdr.NumItems > 0)
			{
				for (int ItemNo = 0; ItemNo < ResultsEx.Hdr.NumItems; ItemNo++)
				{
					if (ResultsEx.Item[ItemNo].OCRCha == 13)
						Msg = Msg + Environment.NewLine;
					else
						Msg = Msg + Convert.ToChar(ResultsEx.Item[ItemNo].OCRCha);
				}
				return true;
			}
			else
			{
				MessageBox.Show("No results returned", "FormatResults", MessageBoxButtons.OK, MessageBoxIcon.Information);
				return false;
			}
		}

		// Convert a bitmap to 1bpp
		private static Bitmap ConvertTo1bpp(Bitmap BMPIn)
		{

			BITMAPINFO bmi = new BITMAPINFO();
			IntPtr hbmIn = BMPIn.GetHbitmap();

			bmi.bmih.biSize			= (uint)Marshal.SizeOf(bmi.bmih);
			bmi.bmih.biWidth		= BMPIn.Width;
			bmi.bmih.biHeight		= BMPIn.Height;
			bmi.bmih.biPlanes		= 1;
			bmi.bmih.biBitCount		= 1;
			bmi.bmih.biCompression	= BI_RGB;
			bmi.bmih.biSizeImage	= (uint)((((BMPIn.Width + 7) & 0xFFFFFFF8) >> 3) * BMPIn.Height);
			bmi.bmih.biXPelsPerMeter= System.Convert.ToInt32(BMPIn.HorizontalResolution * 100 / 2.54);
			bmi.bmih.biYPelsPerMeter= System.Convert.ToInt32(BMPIn.VerticalResolution * 100 / 2.54);
			bmi.bmih.biClrUsed		= 2;
			bmi.bmih.biClrImportant	= 2;
			bmi.cols				= new uint[2]; // see the definition of BITMAPINFO()
			bmi.cols[0]				= 0;
			bmi.cols[1]				= ((uint)(255)) | ((uint)((255) << 8)) | ((uint)((255) << 16));

			IntPtr dummy;
			IntPtr hbm = CreateDIBSection(IntPtr.Zero, ref bmi, DIB_RGB_COLORS, out dummy, IntPtr.Zero, 0);
			
			IntPtr scrnDC = GetDC(IntPtr.Zero);
			IntPtr hDCIn = CreateCompatibleDC(scrnDC);
			
			SelectObject(hDCIn, hbmIn); 
			IntPtr hDC = CreateCompatibleDC(scrnDC);
			SelectObject(hDC, hbm);
			
			BitBlt(hDC, 0, 0, BMPIn.Width, BMPIn.Height, hDCIn, 0, 0, SRCCOPY);
			
			Bitmap BMP = Bitmap.FromHbitmap(hbm);

			DeleteDC(hDCIn);
			DeleteDC(hDC);
			ReleaseDC(IntPtr.Zero, scrnDC);
			DeleteObject(hbmIn);
			DeleteObject(hbm);

			return BMP;
		}

		// Convert a bitmap to a memory mapped file.
		// It does this by constructing a GDI bitmap in a byte array and copying this to a memory mapped file.
		private static IntPtr ConvertBitmapToMMF(Bitmap BMPIn, bool DiscardBitmap, bool ConvertTo1Bit)
		{
			Bitmap		BMP;
			BITMAPINFOHEADER BIH;
			BitmapData	BMPData;
			int			ImageSize;
			byte[]		Bytes;
			GCHandle	BytesGC;
			int			MMFsize;
			int			PalEntries;
			RGBQUAD		rgb;
			int 		Offset;
			IntPtr		MMFhandle = IntPtr.Zero;
			IntPtr		MMFview = IntPtr.Zero;
			IntPtr		RetPtr = IntPtr.Zero;

			if (DiscardBitmap) // can destroy input bitmap
			{
				if (ConvertTo1Bit && BMPIn.PixelFormat != PixelFormat.Format1bppIndexed) 
				{
					BMP = ConvertTo1bpp(BMPIn);
					BMPIn.Dispose();
					BMPIn = null;
				} 
				else 
				{
					BMP = BMPIn;
				}
			} 
			else			  // must keep input bitmap unchanged 
			{
				if (ConvertTo1Bit && BMPIn.PixelFormat != PixelFormat.Format1bppIndexed) 
				{
					BMP = ConvertTo1bpp(BMPIn);
				} 
				else 
				{
					BMP = BMPIn.Clone(new Rectangle(new Point(), BMPIn.Size), BMPIn.PixelFormat);
				}
			}

			// Flip the bitmap (GDI+ bitmap scan lines are top down, GDI are bottom up)
			BMP.RotateFlip(RotateFlipType.RotateNoneFlipY);
			BMPData = BMP.LockBits(new Rectangle(new Point(), BMP.Size), ImageLockMode.ReadOnly, BMP.PixelFormat);
			ImageSize = BMPData.Stride * BMP.Height;
			
			PalEntries = BMP.Palette.Entries.Length;
			
			BIH.biWidth			= BMP.Width;
			BIH.biHeight		= BMP.Height;
			BIH.biPlanes		= 1;
			BIH.biClrImportant	= 0;
			BIH.biCompression	= BI_RGB;
			BIH.biSizeImage		= (uint)ImageSize;
			BIH.biXPelsPerMeter = System.Convert.ToInt32(BMP.HorizontalResolution * 100 / 2.54);
			BIH.biYPelsPerMeter = System.Convert.ToInt32(BMP.VerticalResolution * 100 / 2.54);
			BIH.biBitCount		= 0;	// to avoid "Use of unassigned local variable 'BIH'"
			BIH.biSize			= 0;	// to avoid "Use of unassigned local variable 'BIH'"
			BIH.biClrImportant	= 0;	// to avoid "Use of unassigned local variable 'BIH'"
			
			// Most of these formats are untested and the alpha channel is ignored
			switch ( BMP.PixelFormat )
			{
				case PixelFormat.Format1bppIndexed:
					BIH.biBitCount = 1;
					break;
				case PixelFormat.Format4bppIndexed:
					BIH.biBitCount = 4;
					break;
				case PixelFormat.Format8bppIndexed:
					BIH.biBitCount = 8;
					break;
				case PixelFormat.Format16bppArgb1555:
				case PixelFormat.Format16bppGrayScale:
				case PixelFormat.Format16bppRgb555:
				case PixelFormat.Format16bppRgb565:
					BIH.biBitCount = 16;
					PalEntries = 0;
					break;
				case PixelFormat.Format24bppRgb:
					BIH.biBitCount = 24;
					PalEntries = 0;
					break;
				case  PixelFormat.Format32bppArgb:
				case PixelFormat.Format32bppPArgb:
				case PixelFormat.Format32bppRgb:
					BIH.biBitCount = 32;
					PalEntries = 0;
					break;
			}
			BIH.biClrUsed = (uint)PalEntries;
			BIH.biSize = (uint)Marshal.SizeOf(BIH);

			MMFsize = Marshal.SizeOf(BIH) + PalEntries * Marshal.SizeOf(typeof(RGBQUAD)) + ImageSize;
			Bytes = new byte[MMFsize];

			BytesGC = GCHandle.Alloc(Bytes, GCHandleType.Pinned);
			Marshal.StructureToPtr(BIH, BytesGC.AddrOfPinnedObject(), true);
			Offset = Marshal.SizeOf(BIH);
			rgb.rgbReserved = 0;
			for (int PalEntry = 0; PalEntry <= PalEntries - 1; PalEntry++) 
			{
				rgb.rgbRed   = BMP.Palette.Entries[PalEntry].R;
				rgb.rgbGreen = BMP.Palette.Entries[PalEntry].G;
				rgb.rgbBlue  = BMP.Palette.Entries[PalEntry].B;

				Marshal.StructureToPtr(rgb, Marshal.UnsafeAddrOfPinnedArrayElement(Bytes, Offset), false);
				Offset = Offset + Marshal.SizeOf(rgb);
			}
			BytesGC.Free();
			Marshal.Copy(BMPData.Scan0, Bytes, Offset, ImageSize);
			BMP.UnlockBits(BMPData);
			BMPData = null;
			BMP.Dispose();
			BMP = null;
			MMFhandle = CreateFileMappingMy(0xFFFFFFFF, 0, PAGE_READWRITE, 0, (uint)MMFsize, 0);
			if (!(MMFhandle.Equals(IntPtr.Zero))) 
			{
				MMFview = MapViewOfFileMy(MMFhandle, FILE_MAP_WRITE, 0, 0, 0);
				if (MMFview.Equals(IntPtr.Zero)) 
				{
					CloseHandle(MMFhandle);
				} 
				else 
				{
					Marshal.Copy(Bytes, 0, MMFview, MMFsize);
					UnmapViewOfFileMy(MMFview);
					RetPtr = MMFhandle;
				}
			}

			Bytes = null;

			if (RetPtr.Equals(IntPtr.Zero)) 
			{
				MessageBox.Show("Failed to convert bitmap", "ConvertBitmapToMMF", MessageBoxButtons.OK, MessageBoxIcon.Stop);
			}
			return RetPtr;
		}

		private static IntPtr ConvertBitmapToMMF(Bitmap BMPIn, bool DiscardBitmap) 
		{
			return ConvertBitmapToMMF(BMPIn, DiscardBitmap, true);
		}

		private static IntPtr ConvertBitmapToMMF(Bitmap BMPIn) 
		{
			return ConvertBitmapToMMF(BMPIn, true, true);
		}

		// Convert a bitmap to a memory mapped file
		// (Same as ConvertBitmapToMMF but uses CopyMemory to avoid using a byte array)
		private static IntPtr ConvertBitmapToMMF2(Bitmap BMPIn, bool DiscardBitmap, bool ConvertTo1Bit)
		{
			Bitmap		BMP;
			BITMAPINFOHEADER BIH;
			BitmapData	BMPData;
			int			ImageSize;
			int			MMFsize;
			int			PalEntries;
			RGBQUAD		rgb;
			GCHandle	rgbGC;
			int			Offset;
			IntPtr		MMFhandle = IntPtr.Zero;
			IntPtr		MMFview = IntPtr.Zero;
			IntPtr		RetPtr = IntPtr.Zero;

			if (DiscardBitmap) // can destroy input bitmap
			{
				if (ConvertTo1Bit && BMPIn.PixelFormat != PixelFormat.Format1bppIndexed) 
				{
					BMP = ConvertTo1bpp(BMPIn);
					BMPIn.Dispose();
					BMPIn = null;
				} 
				else 
				{
					BMP = BMPIn;
				}
			} 
			else			  // must keep input bitmap unchanged 
			{
				if (ConvertTo1Bit && BMPIn.PixelFormat != PixelFormat.Format1bppIndexed) 
				{
					BMP = ConvertTo1bpp(BMPIn);
				} 
				else 
				{
					BMP = BMPIn.Clone(new Rectangle(new Point(), BMPIn.Size), BMPIn.PixelFormat);
				}
			}


			// Flip the bitmap (GDI+ bitmap scan lines are top down, GDI are bottom up)
			BMP.RotateFlip(RotateFlipType.RotateNoneFlipY);
			
			BMPData = BMP.LockBits(new Rectangle(new Point(), BMP.Size), ImageLockMode.ReadOnly, BMP.PixelFormat);
			ImageSize = BMPData.Stride * BMP.Height;
			
			PalEntries = BMP.Palette.Entries.Length;
			
			BIH.biWidth			= BMP.Width;
			BIH.biHeight		= BMP.Height;
			BIH.biPlanes		= 1;
			BIH.biClrImportant	= 0;
			BIH.biCompression	= BI_RGB;
			BIH.biSizeImage		= (uint)ImageSize;
			BIH.biXPelsPerMeter	= System.Convert.ToInt32(BMP.HorizontalResolution * 100 / 2.54);
			BIH.biYPelsPerMeter = System.Convert.ToInt32(BMP.VerticalResolution * 100 / 2.54);
			BIH.biBitCount		= 0;	// to avoid "Use of unassigned local variable 'BIH'"
			BIH.biSize			= 0;	// to avoid "Use of unassigned local variable 'BIH'"
			BIH.biClrImportant	= 0;	// to avoid "Use of unassigned local variable 'BIH'"

			// Most of these formats are untested and the alpha channel is ignored
			switch ( BMP.PixelFormat )
			{
				case PixelFormat.Format1bppIndexed:
					BIH.biBitCount = 1;
					break;
				case PixelFormat.Format4bppIndexed:
					BIH.biBitCount = 4;
					break;
				case PixelFormat.Format8bppIndexed:
					BIH.biBitCount = 8;
					break;
				case PixelFormat.Format16bppArgb1555:
				case PixelFormat.Format16bppGrayScale:
				case PixelFormat.Format16bppRgb555:
				case PixelFormat.Format16bppRgb565:
					BIH.biBitCount = 16;
					PalEntries = 0;
					break;
				case PixelFormat.Format24bppRgb:
					BIH.biBitCount = 24;
					PalEntries = 0;
					break;
				case  PixelFormat.Format32bppArgb:
				case PixelFormat.Format32bppPArgb:
				case PixelFormat.Format32bppRgb:
					BIH.biBitCount = 32;
					PalEntries = 0;
					break;
			}
			BIH.biClrUsed = (uint)PalEntries;
			BIH.biSize = (uint)Marshal.SizeOf(BIH);

			MMFsize = Marshal.SizeOf(BIH) + PalEntries * Marshal.SizeOf(typeof(RGBQUAD)) + ImageSize;
			
			MMFhandle = CreateFileMappingMy(0xFFFFFFFF, 0, PAGE_READWRITE, 0, (uint)MMFsize, 0);
			if (!(MMFhandle.Equals(IntPtr.Zero))) 
			{
				MMFview = MapViewOfFileMy(MMFhandle, FILE_MAP_WRITE, 0, 0, 0);
				if (MMFview.Equals(IntPtr.Zero)) 
				{
					CloseHandle(MMFhandle);
				} 
				else 
				{
					Marshal.StructureToPtr(BIH, MMFview, true);
					Offset = MMFview.ToInt32() + Marshal.SizeOf(BIH);
					rgb.rgbReserved = 0;
					for (int PalEntry = 0; PalEntry <= PalEntries - 1; PalEntry++) 
					{
						rgb.rgbRed   = BMP.Palette.Entries[PalEntry].R;
						rgb.rgbGreen = BMP.Palette.Entries[PalEntry].G;
						rgb.rgbBlue  = BMP.Palette.Entries[PalEntry].B;

						rgbGC = GCHandle.Alloc(rgb, GCHandleType.Pinned);
						CopyMemory((uint)Offset, rgbGC.AddrOfPinnedObject(), (uint)Marshal.SizeOf(rgb));
						rgbGC.Free();
						Offset = Offset + Marshal.SizeOf(rgb);
					}
					CopyMemory((uint)Offset, BMPData.Scan0, (uint)ImageSize);
					UnmapViewOfFileMy(MMFview);
					RetPtr = MMFhandle;
				}
			}
			BMP.UnlockBits(BMPData);
			BMPData = null;
			BMP.Dispose();
			BMP = null;

			if (RetPtr.Equals(IntPtr.Zero)) 
			{
				MessageBox.Show("Failed to convert bitmap", "ConvertBitmapToMMF2", MessageBoxButtons.OK, MessageBoxIcon.Stop);
			}
			return RetPtr;
		}

		private static IntPtr ConvertBitmapToMMF2(Bitmap BMPIn, bool DiscardBitmap) 
		{
			return ConvertBitmapToMMF2(BMPIn, DiscardBitmap, true);
		}

		private static IntPtr ConvertBitmapToMMF2(Bitmap BMPIn) 
		{
			return ConvertBitmapToMMF2(BMPIn, true, true);
		}

		// Convert a global memory block to a bitmap
		private static Bitmap ConvertMemoryBlockToBitmap(IntPtr hMem)
		{
			Bitmap BMP = null;

			BITMAPINFOHEADER BIH = new BITMAPINFOHEADER();
			IntPtr		bihPtr;
			IntPtr		dataPtr;
			IntPtr		palPtr;
			uint		HdrSize;
			PixelFormat	PixFormat = PixelFormat.Format1bppIndexed;
			uint		PalEntries = 0;
			RGBQUAD		rgb = new RGBQUAD();
	
			bihPtr = GlobalLock(hMem);

			if (!bihPtr.Equals(IntPtr.Zero))
			{
				BIH = ((BITMAPINFOHEADER)(Marshal.PtrToStructure(bihPtr, typeof(BITMAPINFOHEADER))));
				HdrSize = BIH.biSize;
				palPtr = (IntPtr)(bihPtr.ToInt32() + HdrSize);

				// Most of these formats are untested
				switch (BIH.biBitCount)
				{
					case 1:
						HdrSize += (uint)(2 * Marshal.SizeOf(rgb));
						PixFormat = PixelFormat.Format1bppIndexed;
						PalEntries = 2;
						break;
					case 4:
						HdrSize += (uint)(16 * Marshal.SizeOf(rgb));
						PixFormat = PixelFormat.Format4bppIndexed;
						PalEntries = BIH.biClrUsed;
						break;
					case 8:
						HdrSize += (uint)(256 * Marshal.SizeOf(rgb));
						PixFormat = PixelFormat.Format8bppIndexed;
						PalEntries = BIH.biClrUsed;
						break;
					case 16:
						// Account for the 3 DWORD colour mask
						if (BIH.biCompression == BI_BITFIELDS)
							HdrSize += 12;
						PixFormat = PixelFormat.Format16bppRgb555;
						PalEntries = 0;
						break;
					case 24:
						PixFormat = PixelFormat.Format24bppRgb;
						PalEntries = 0;
						break;
					case 32:
						// Account for the 3 DWORD colour mask
						if (BIH.biCompression == BI_BITFIELDS)
							HdrSize += 12;
						PixFormat = PixelFormat.Format32bppRgb;
						PalEntries = 0;
						break;
					default:
						break;
				}

				dataPtr = (IntPtr)(bihPtr.ToInt32() + HdrSize);
				BMP = new Bitmap(BIH.biWidth,Math.Abs(BIH.biHeight), PixFormat);
				if ( PalEntries > 0 )
				{
					palPtr = (IntPtr)(bihPtr.ToInt32() + BIH.biSize);
					ColorPalette Pal = BMP.Palette;
					for (int PalEntry = 0; PalEntry < PalEntries; PalEntry++) 
					{
						rgb = ((RGBQUAD)(Marshal.PtrToStructure(palPtr, typeof(RGBQUAD))));
						Pal.Entries[PalEntry] = Color.FromArgb(rgb.rgbRed, rgb.rgbGreen, rgb.rgbBlue);
						palPtr = (IntPtr)(palPtr.ToInt32() + Marshal.SizeOf(rgb));
					}
					BMP.Palette = Pal;
				}
				BitmapData BMPData = BMP.LockBits(new Rectangle(new Point(), BMP.Size), ImageLockMode.ReadWrite, PixFormat);
				CopyMemory((uint)BMPData.Scan0.ToInt32(), dataPtr, (uint)(BMPData.Stride * BMP.Height));
				BMP.UnlockBits(BMPData);
				// Flip the bitmap (GDI+ bitmap scan lines are top down, GDI are bottom up)
				BMP.RotateFlip(RotateFlipType.RotateNoneFlipY);

				// Reset the resolutions
				BMP.SetResolution((float)((int)(BIH.biXPelsPerMeter * 2.54 / 100 + 0.5)), (float)((int)(BIH.biYPelsPerMeter * 2.54 / 100 + 0.5)));
				GlobalUnlock(hMem);
			}

			return BMP;
		}
	}
}
