Attribute VB_Name = "modMain"
' Transym OCR Demonstration program
'
' THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
' EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
' WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
'
' This program demonstrates calling TOCR version 4.0 from VB6
'
' Copyright (C) 2012 Transym Computer Services Ltd.
'
'
' TOCR4.0DemoVB6 Issue1

Option Explicit

 Const BITMAP_OK = 0                ' all OK
 Const BITMAP_ERRS = 1              ' VB error encountered
 Const BITMAP_FAIL = 2              ' file is a bitmap but it failed to load
 Const BITMAP_NOTBITMAP = 3         ' file is not a bitmap

 Type BMPINFO ' just the info on bitmaps that I'm interested in
    hBmp            As Long         ' handle to bitmap
    Width           As Long         ' pixel width of bitmap
    Height          As Long         ' pixel height of bitmap
    XPelsPerMeter   As Long         ' X pixels per metre
    YPelsPerMeter   As Long         ' Y pixels per metre
End Type ' BMPINFO

 Const LEN_BITMAPCOREHEADER = 12    ' size in bytes of BITMAPCOREHEADER
 Const LEN_BITMAPINFOHEADER = 40    ' size in bytes of BITMAPINFOHEADER
 Const LEN_RGBQUAD = 4              ' size in bytes of RGBQUAD
 
 Const ERRCANTFINDDLLENTRYPOINT = 453 ' Can't find DLL entry point

Const SAMPLE_TIFF_FILE = "Sample.tif"
Const SAMPLE_BMP_FILE = "Sample.bmp"

'---------------------------------------------------------------------------
' Application start up module
'
Sub Main()


Call Example1   ' demonstrates how to OCR a single file
Call Example2   ' demonstrates how to OCR multiple files
Call Example3   ' demonstrates how to OCR a bitmap using a memory mapped file created by TOCR
Call Example4   ' demonstrates how to OCR a bitmap using a memory mapped file created here
Call Example5   ' retrieves information on job slot usage
Call Example6   ' retrieves information on job slots
Call Example7   ' gets images from a TWAIN compatible device
Call Example8   ' demonstrates TOCRSetConfig and TOCRGetConfig

End

End Sub ' Main

'---------------------------------------------------------------------------
' Demonstrates how to OCR a single file
'
Sub Example1()

Dim Status              As Long
Dim JobInfo2            As TOCRJOBINFO2
Dim JobNo               As Long
Dim Results             As TOCRRESULTS
Dim Answer              As String

TOCRSetConfig TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_MSGBOX

JobInfo2.InputFile = SAMPLE_TIFF_FILE
JobInfo2.JobType = TOCRJOBTYPE_TIFFFILE

'or

JobInfo2.InputFile = SAMPLE_BMP_FILE
JobInfo2.JobType = TOCRJOBTYPE_DIBFILE

Status = TOCRInitialise(JobNo)
If Status = TOCR_OK Then
    'If OCRPoll(JobNo, JobInfo2) Then
    If OCRWait(JobNo, JobInfo2) Then
        If GetResults(JobNo, Results) Then
            If FormatResults(Results, Answer) Then
                MsgBox Answer, , "Example 1"
            End If
        End If
    End If
    TOCRShutdown JobNo
End If

End Sub ' Example1

'---------------------------------------------------------------------------
' Demonstrates how to OCR multiple files
'
Sub Example2()

Dim Status              As Long
Dim JobNo               As Long
Dim JobInfo2            As TOCRJOBINFO2
Dim Results             As TOCRRESULTS
Dim CountDone           As Long

TOCRSetConfig TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_MSGBOX

Status = TOCRInitialise(JobNo)
If Status = TOCR_OK Then

    ' 1st file
    JobInfo2.InputFile = SAMPLE_TIFF_FILE
    JobInfo2.JobType = TOCRJOBTYPE_TIFFFILE
    If OCRWait(JobNo, JobInfo2) Then
        If GetResults(JobNo, Results) Then
            CountDone = CountDone + 1
        End If
    End If

    ' 2nd file
    JobInfo2.InputFile = SAMPLE_BMP_FILE
    JobInfo2.JobType = TOCRJOBTYPE_DIBFILE
    If OCRWait(JobNo, JobInfo2) Then
        If GetResults(JobNo, Results) Then
            CountDone = CountDone + 1
        End If
    End If
    TOCRShutdown (JobNo)
End If

MsgBox CStr(CountDone) & " of 2 jobs done", vbInformation, "Example 2"

End Sub ' Example2

'---------------------------------------------------------------------------
' Demonstrate how to OCR a bitmap using a memory mapped file created by TOCR.
'
Sub Example3()

Dim Status              As Long
Dim JobInfo2            As TOCRJOBINFO2
Dim JobNo               As Long
Dim Results             As TOCRRESULTS
Dim Answer              As String
Dim hFile               As Long

TOCRSetConfig TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_MSGBOX

Status = TOCRInitialise(JobNo)
If Status = TOCR_OK Then
    If TOCRConvertFormat(JobNo, ByVal SAMPLE_TIFF_FILE, TOCRCONVERTFORMAT_TIFFFILE, hFile, TOCRCONVERTFORMAT_MMFILEHANDLE, 0) = TOCR_OK Then
        
        JobInfo2.JobType = TOCRJOBTYPE_MMFILEHANDLE
        
        JobInfo2.hMMF = hFile
                
        If OCRWait(JobNo, JobInfo2) Then
           If GetResults(JobNo, Results) Then
               If FormatResults(Results, Answer) Then
                   MsgBox Answer, vbInformation, "Example 3"
               End If
           End If
        End If
        
        CloseHandle (hFile)
    End If
    
    TOCRShutdown JobNo
End If

End Sub ' Example3

'---------------------------------------------------------------------------
' Demonstrate how to OCR a bitmap using a memory mapped file created here.
' In this example the bitmap is simply read from a file but it could easily be
' one in memory that you have processed in some way.
'
Sub Example4()

Dim Status              As Long
Dim JobInfo2            As TOCRJOBINFO2
Dim JobNo               As Long
Dim Results             As TOCRRESULTS
Dim Answer              As String
Dim BI                  As BMPINFO
Dim hFile               As Long

TOCRSetConfig TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_MSGBOX

Status = TOCRInitialise(JobNo)
If Status = TOCR_OK Then

    ' Get a bitmap into memory
    
    If GetBitmap(BI, SAMPLE_BMP_FILE) = BITMAP_OK Then
        If SaveMonoBitmapToMMFile(BI, hFile) Then
        
            JobInfo2.JobType = TOCRJOBTYPE_MMFILEHANDLE
            
            JobInfo2.hMMF = hFile
            
            If OCRWait(JobNo, JobInfo2) Then
                If GetResults(JobNo, Results) Then
                    If FormatResults(Results, Answer) Then
                        MsgBox Answer, vbInformation, "Example 3"
                    End If
                End If
            End If
            
            CloseHandle hFile
        End If
        If BI.hBmp Then DeleteObject BI.hBmp
    End If
    
    TOCRShutdown JobNo
End If

End Sub ' Example4

'---------------------------------------------------------------------------
' Retrieve information on job slot usage.
'
Sub Example5()

Dim NumSlots            As Long
Dim SlotUse()           As Long
Dim Msg                 As String
Dim SlotNo              As Long

TOCRSetConfig TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_MSGBOX

' uncomment to see effect on job slot use
'Dim JobNo               As Long
'If Not TOCRInitialise(JobNo) = TOCR_OK Then End

NumSlots = TOCRGetJobDBInfo(ByVal 0&)
If NumSlots > 0 Then
    ReDim SlotUse(0 To NumSlots - 1) As Long
    If TOCRGetJobDBInfo(SlotUse(0)) = TOCR_OK Then
        Msg = "Slot usage is:" & vbCrLf
        For SlotNo = 0 To NumSlots - 1
            Msg = Msg & vbCrLf & "Slot" & Str$(SlotNo) & " is "
            Select Case SlotUse(SlotNo)
                Case TOCRJOBSLOT_FREE
                    Msg = Msg & "free"
                Case TOCRJOBSLOT_OWNEDBYYOU
                    Msg = Msg & "owned by you"
                Case TOCRJOBSLOT_BLOCKEDBYYOU
                    Msg = Msg & "blocked by you"
                Case TOCRJOBSLOT_OWNEDBYOTHER
                    Msg = Msg & "owned by another process"
                Case TOCRJOBSLOT_BLOCKEDBYOTHER
                    Msg = Msg & "blocked by another process"
            End Select
        Next SlotNo
        MsgBox Msg, vbInformation, "Example 5"
    Else
        MsgBox "Failed to get job slot information", vbExclamation, "Example 5"
    End If
Else
    MsgBox "Failed to get number of job slots", vbExclamation, "Example 5"
End If

'TOCRShutdown JobNo

End Sub ' Example5

'---------------------------------------------------------------------------
' Retrieve information on job slots.
'
Sub Example6()

Dim NumSlots            As Long
Dim SlotUse()           As Long
Dim Msg                 As String
Dim SlotNo              As Long
Dim Volume              As Long
Dim Time                As Long
Dim Remaining           As Long
Dim Features            As Long
Dim Licence             As String

NumSlots = TOCRGetJobDBInfo(ByVal 0&)
If NumSlots > 0 Then
    Msg = "Slot usage is" & vbCrLf
    For SlotNo = 0 To NumSlots - 1
        Msg = Msg & vbCrLf & "Slot" & Str$(SlotNo)
        Licence = Space$(19)
        If TOCRGetLicenceInfoEx(SlotNo, Licence, Volume, Time, Remaining, Features) = TOCR_OK Then
            Msg = Msg & " " & Licence
            Select Case Features
                Case TOCRLICENCE_STANDARD
                    Msg = Msg & " STANDARD licence"
                Case TOCRLICENCE_EURO
                    If Licence = "5AD4-1D96-F632-8912" Then
                        Msg = Msg & " EURO TRIAL licence"
                    Else
                        Msg = Msg & " EURO licence"
                    End If
                Case TOCRLICENCE_EUROUPGRADE
                    Msg = Msg & " EURO UPGRADE licence"
                Case TOCRLICENCE_V3SE
                    If Licence = "2E72-2B35-643A-0851" Then
                        Msg = Msg & " V3 TRIAL licence"
                    Else
                        Msg = Msg & " V3 licence"
                    End If
                Case TOCRLICENCE_V3SEUPGRADE
                    Msg = Msg & " V1/2 UPGRADE to V3 SE licence"
                Case TOCRLICENCE_V3PRO
                    Msg = Msg & " V3 Pro/V4 licence"
                Case TOCRLICENCE_V3PROUPGRADE
                    Msg = Msg & " V1/2 UPGRADE to V3 Pro/V4 licence"
                Case TOCRLICENCE_V3SEPROUPGRADE
                    Msg = Msg & " V3 SE UPGRADE to V3 Pro/V4 licence"
            End Select
            If Volume <> 0 Or Time <> 0 Then
                Msg = Msg & Str$(Remaining)
                If Time <> 0 Then
                    Msg = Msg & " days"
                Else
                    Msg = Msg & " A4 pages"
                End If
                Msg = Msg & " remaining on licence"
            End If
        End If
    Next SlotNo
    MsgBox Msg, vbInformation, "Example 6"
Else
    MsgBox "Failed to get number of job slots", vbExclamation, "Example 6"
End If ' NumSlots > 0

End Sub ' Example6

'---------------------------------------------------------------------------
' Get images from a TWAIN compatible device
'
Sub Example7()

Dim NumImages           As Long         ' number of images acquired
Dim BI                  As BMPINFO
Dim hDIB()              As Long         ' handles to memory blocks holding images
Dim ImgCnt              As Long         ' counter
Dim DIBNo               As Long         ' loop counter


On Error Resume Next
TOCRTWAINAcquire NumImages
If Err = ERRCANTFINDDLLENTRYPOINT Then
    MsgBox "This version of TOCR DLL does not support TWAIN", vbExclamation
    Exit Sub
End If
On Error GoTo 0

If NumImages Then
    ReDim hDIB(0 To NumImages - 1)
    
    ImgCnt = 0
    
    TOCRTWAINGetImages hDIB(0)
    
    ' Convert the memory pointers to bitmap handles
    
    BI.hBmp = 0
    For DIBNo = 0 To NumImages - 1
        If GetMonoBitmapFromDIB(BI, hDIB(DIBNo)) Then
            ImgCnt = ImgCnt + 1
        End If
        
        ' Free memory as you go along
        
        If hDIB(DIBNo) Then GlobalFree hDIB(DIBNo)
        hDIB(DIBNo) = 0
        
        If BI.hBmp Then DeleteObject BI.hBmp
        BI.hBmp = 0
    Next DIBNo
    MsgBox CStr(ImgCnt) & " images acquired", vbInformation, "Example 7"
Else
    MsgBox "No images acquired", vbInformation, "Example 7"
End If


End Sub ' Example7

'---------------------------------------------------------------------------
' Demonstrate TOCRSetConfig and TOCRGetConfig
'
Sub Example8()

Dim JobNo               As Long
Dim Answer              As String
Dim Value               As Long

Answer = Space(250)

' Override the INI file settings for all new jobs
TOCRSetConfig TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_MSGBOX
TOCRSetConfig TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_SRV_ERRORMODE, TOCRERRORMODE_MSGBOX

TOCRGetConfigStr TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_LOGFILE, Answer
MsgBox "Default Log file name = " & Answer, vbInformation, "Example 8"

TOCRSetConfigStr TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_LOGFILE, "Loggederrs.lis"
TOCRGetConfigStr TOCRCONFIG_DEFAULTJOB, TOCRCONFIG_LOGFILE, Answer
MsgBox "New default Log file name = " & Answer, vbInformation, "Example 8"

TOCRInitialise (JobNo)
TOCRSetConfig JobNo, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_NONE

TOCRGetConfig JobNo, TOCRCONFIG_DLL_ERRORMODE, Value
MsgBox "Job DLL error mode = " & CStr(Value), vbInformation, "Example 8"

TOCRGetConfig JobNo, TOCRCONFIG_SRV_ERRORMODE, Value
MsgBox "Job Service error mode = " & CStr(Value), vbInformation, "Example 8"

TOCRGetConfigStr JobNo, TOCRCONFIG_LOGFILE, Answer
MsgBox "Job Log file name = " & Answer, vbInformation, "Example 8"

' Cause an error - then check Loggederrs.lis
TOCRSetConfig JobNo, TOCRCONFIG_DLL_ERRORMODE, TOCRERRORMODE_LOG
TOCRSetConfig JobNo, 1000, TOCRERRORMODE_LOG

End Sub ' Example8

' Wait for the engine to complete
Private Function OCRWait(ByVal JobNo As Integer, JobInfo2 As TOCRJOBINFO2) As Boolean

Dim Status          As Long
Dim JobStatus       As Long
Dim Msg             As String
Dim ErrorMode       As Long


Status = TOCRDoJob2(JobNo, JobInfo2)
If Status = TOCR_OK Then
    Status = TOCRWaitForJob(JobNo, JobStatus)
End If

If Status = TOCR_OK And JobStatus = TOCRJOBSTATUS_DONE Then
    OCRWait = True
Else
    OCRWait = False

    ' If something hass gone wrong display a message
    ' (Check that the OCR engine hasn't already displayed a message)
    TOCRGetConfig JobNo, TOCRCONFIG_DLL_ERRORMODE, ErrorMode
    If ErrorMode = TOCRERRORMODE_NONE Then
        Msg = Space$(TOCRJOBMSGLENGTH)
        TOCRGetJobStatusMsg JobNo, Msg
        MsgBox Msg, vbCritical, "OCRWait"
    End If
End If

End Function

' Wait for the engine to complete by polling
Private Function OCRPoll(ByVal JobNo As Integer, JobInfo2 As TOCRJOBINFO2) As Boolean

Dim Status                  As Long
Dim JobStatus               As Long
Dim Msg                     As String
Dim ErrorMode               As Long
Dim Progress                As Single
Dim AutoOrientation         As Long

Status = TOCRDoJob2(JobNo, JobInfo2)
If Status = TOCR_OK Then
    Do
        'Status = TOCRGetJobStatus(JobNo, JobStatus)
        Status = TOCRGetJobStatusEx(JobNo, JobStatus, Progress, AutoOrientation)

        ' Do something whilst the OCR engine runs
        DoEvents: Sleep (100): DoEvents
        Debug.Print ("Progress" & Str$(Int(Progress * 100)) & "%")

    Loop While Status = TOCR_OK And JobStatus = TOCRJOBSTATUS_BUSY
End If

If Status = TOCR_OK And JobStatus = TOCRJOBSTATUS_DONE Then
    OCRPoll = True
Else
    OCRPoll = False

    ' If something hass gone wrong display a message
    ' (Check that the OCR engine hasn't already displayed a message)
    TOCRGetConfig JobNo, TOCRCONFIG_DLL_ERRORMODE, ErrorMode
    If ErrorMode = TOCRERRORMODE_NONE Then
        Msg = Space$(TOCRJOBMSGLENGTH)
        TOCRGetJobStatusMsg JobNo, Msg
        MsgBox Msg, vbCritical, "OCRPoll"
    End If
End If

End Function


'---------------------------------------------------------------------------
' Find the number of colours used in a DIB.  This allows a logical palette
' to be created
'
Private Function DIBNumColours(bmih As BITMAPINFOHEADER, ByVal IsCoreHeader As Boolean)

Dim NumColours          As Long         ' number of colours used in the DIB

If IsCoreHeader Then
    NumColours = 2 ^ bmih.biBitCount
Else
    If bmih.biClrUsed <> 0 Then
        NumColours = bmih.biClrUsed
    Else
        NumColours = 0
        If bmih.biBitCount = 1 Then NumColours = 2
        If bmih.biBitCount = 4 Then NumColours = 16
        If bmih.biBitCount = 8 Then NumColours = 256
    End If
End If

DIBNumColours = NumColours

End Function ' DIBNumColours

'---------------------------------------------------------------------------
' Create a logical palette for a DIB
'
Private Function DIBPalette(ByVal NumColours As Long, bmib() As Byte, ByVal IsCoreHeader As Boolean) As Long

Dim Pal                 As LOGPALETTE   ' logical palette
Dim hPal                As Long         ' handle to Pal
Dim PalEntry()          As PALETTEENTRY ' palette entry in Pal
Dim RGB3()              As RGBTRIPLE    ' an RGB triple colour value
Dim RGB4()              As RGBQUAD      ' an RGB quad value
Dim hMem                As Long         ' handle to a memory block
Dim lpMem               As Long         ' pointer to a memory block
Dim Clrno               As Long         ' loop counter for colours

DIBPalette = 0
If NumColours = 0 Then Exit Function

ReDim PalEntry(0 To NumColours - 1) As PALETTEENTRY

' Load PalEntry from the DIB

If IsCoreHeader Then
    ReDim RGB3(0 To NumColours - 1) As RGBTRIPLE
    
    CopyMemory RGB3(0), bmib(LEN_BITMAPCOREHEADER), NumColours * Len(RGB3(0))
    
    For Clrno = 0 To NumColours - 1
        PalEntry(Clrno).peRed = RGB3(Clrno).rgbtRed
        PalEntry(Clrno).peBlue = RGB3(Clrno).rgbtBlue
        PalEntry(Clrno).peGreen = RGB3(Clrno).rgbtGreen
        PalEntry(Clrno).peFlags = 0
    Next Clrno
    
    Erase RGB3
Else
    ReDim RGB4(0 To NumColours - 1) As RGBQUAD
    
    CopyMemory RGB4(0), bmib(LEN_BITMAPINFOHEADER), NumColours * Len(RGB4(0))
    
    For Clrno = 0 To NumColours - 1
        PalEntry(Clrno).peRed = RGB4(Clrno).rgbRed
        PalEntry(Clrno).peBlue = RGB4(Clrno).rgbBlue
        PalEntry(Clrno).peGreen = RGB4(Clrno).rgbGreen
        PalEntry(Clrno).peFlags = 0
    Next Clrno
    
    Erase RGB4
End If

Pal.PALVERSION = PALVERSION
Pal.palNumEntries = NumColours

' Copy Pal to a memory block and return a handle to it

hMem = GlobalAlloc(GHND, Len(Pal) + Len(PalEntry(0)) * NumColours)
If hMem = 0 Then Exit Function

lpMem = GlobalLock(hMem)
If lpMem = 0 Then
    GlobalFree (hMem)
    Exit Function
End If

CopyMemory ByVal lpMem, Pal, Len(Pal)
CopyMemory ByVal lpMem + 4, PalEntry(0), Len(PalEntry(0)) * NumColours

hPal = CreatePaletteMy(lpMem)
GlobalUnlock (hMem)
GlobalFree (hMem)

DIBPalette = hPal

End Function ' DIBPalette

'---------------------------------------------------------------------------
' Convert a DIB to a mono bitmap
'
Private Function DIBtoMonoBitmap(bmi() As Byte, Data() As Byte, BI As BMPINFO) As Long

Dim bmih                As BITMAPINFOHEADER
Dim bmch                As BITMAPCOREHEADER
Dim hMonoBmp            As Long         ' bitmap handle to mono
Dim hClrBmp             As Long         ' bitmap handle to colour
Dim result              As Long         ' API return status
Dim hDCMem              As Long         ' handle to a memory device context
Dim DataAddr            As Long         ' address of DIB section data
Dim IsCoreHeader        As Boolean      ' flag if bitmap header is BITMAPCOREHEADER
Dim NumColours          As Long         ' number of colours in the DIB palette
Dim hPal                As Long         ' handle to a palette
Dim hPalOld             As Long         ' handle to a palette

DIBtoMonoBitmap = BITMAP_NOTBITMAP

hMonoBmp = 0
hClrBmp = 0
hPalOld = 0
hPal = 0

On Error GoTo DMBErrs

' See what type of header it has

CopyMemory bmih, bmi(0), 4
If bmih.biSize = Len(bmch) Then
    IsCoreHeader = True
    CopyMemory bmch, bmi(0), Len(bmch)
    
    bmih.biBitCount = bmch.bcBitCount
    bmih.biHeight = bmch.bcHeight
    bmih.biPlanes = bmch.bcPlanes
    bmih.biWidth = bmch.bcWidth
    bmih.biCompression = BI_RGB
    bmih.biXPelsPerMeter = 3780 ' assumed values
    bmih.biYPelsPerMeter = 3780 ' assumed values
Else
    IsCoreHeader = False
    CopyMemory bmih, bmi(0), Len(bmih)
End If

' Validate BITMAPINFOHEADER

If bmih.biBitCount <> 1 And bmih.biBitCount <> 4 And bmih.biBitCount <> 8 And _
    bmih.biBitCount <> 16 And bmih.biBitCount <> 24 And bmih.biBitCount <> 32 Then Exit Function
If bmih.biWidth = 0 Or bmih.biHeight = 0 Then Exit Function
If bmih.biCompression <> BI_RGB And bmih.biCompression <> BI_RLE4 And _
     bmih.biCompression <> BI_RLE8 And bmih.biCompression <> BI_BITFIELDS Then Exit Function
If (bmih.biCompression = BI_RLE4 And bmih.biBitCount <> 4) Or _
    (bmih.biCompression = BI_RLE8 And bmih.biBitCount <> 8) Then Exit Function
If bmih.biPlanes <> 1 Then Exit Function

' At this point think it's a bitmap - but may fail to load it

DIBtoMonoBitmap = BITMAP_FAIL

' Create a mono bitmap as this is what the Service process will use
'
' If you implement a better way to do this then when you OCR you should
' pass a bitmap and not the filename to the OCR service process because
' the service process mimics the behaviour below (change JobType from
' TOCRJOBTYPE_DIBFILE to TOCRJOBTYPE_DIBCLIPBOARD in frmViewer.GetFile).

result = 0
hDCMem = GetDC(0&)
If hDCMem Then

    ' Create a logical palette from the DIB if required and load
    
    If GetDeviceCaps(hDCMem, RASTERCAPS) And RC_PALETTE Then
        NumColours = DIBNumColours(bmih, IsCoreHeader)
        If NumColours Then
            hPal = DIBPalette(NumColours, bmi(), IsCoreHeader)
            If hPal Then
                hPalOld = SelectPalette(hDCMem, hPal, False)
                RealizePalette hDCMem
            End If
        End If
    End If
    
    ' Create a colour bitmap for the data
    ' CreateDIBSection fails for run length encoded files
    If bmih.biCompression = BI_RLE4 Or bmih.biCompression = BI_RLE8 Then
        hClrBmp = CreateDIBitmapMy(hDCMem, bmi(0), 0, ByVal 0&, ByVal 0&, DIB_PAL_COLORS)
    Else
        hClrBmp = CreateDIBSectionMy(hDCMem, bmi(0), DIB_RGB_COLORS, DataAddr, 0&, 0&)
    End If
    If hClrBmp Then
        If SetDIBitsMy(hDCMem, hClrBmp, 0, Abs(bmih.biHeight), Data(0), bmi(0), DIB_RGB_COLORS) Then
            GdiFlush

            ' Create mono bitmap info

            bmih.biSize = Len(bmih)
            bmih.biBitCount = 1
            bmih.biClrImportant = 0
            bmih.biClrUsed = 2
            bmih.biCompression = BI_RGB
            bmih.biSizeImage = Int((bmih.biWidth + 31&) / 32) * 4& * Abs(bmih.biHeight)

            ReDim Data(0 To bmih.biSizeImage - 1)
            
            ' Ensure bmi is long enough for the header and colour table

            ReDim bmi(0 To Len(bmih) + LEN_RGBQUAD * 2 - 1)

            ' Load the new header into bmi

            CopyMemory bmi(0), bmih, Len(bmih)

            If GetDIBitsMy(hDCMem, hClrBmp, 0, Abs(bmih.biHeight), Data(0), bmi(0), DIB_RGB_COLORS) Then

                ' Save some more space

                DeleteObject hClrBmp
                hClrBmp = 0

                hMonoBmp = CreateBitmap(bmih.biWidth, Abs(bmih.biHeight), 1, 1, ByVal 0&)
                If hMonoBmp Then
                    result = SetDIBitsMy(hDCMem, hMonoBmp, 0, Abs(bmih.biHeight), Data(0), bmi(0), DIB_RGB_COLORS)
                    GdiFlush
                End If
            End If
            
        End If ' SetDIBitsMy
    End If ' hClrBmp
    
    ' Clean up
    
    If result = 0 And hMonoBmp Then
        DeleteObject hMonoBmp
        hMonoBmp = 0
    End If
    
    If hClrBmp Then
        DeleteObject hClrBmp
        hClrBmp = 0
    End If
    
    If hPalOld Then
        hPal = SelectPalette(hDCMem, hPalOld, False)
        hPalOld = 0
        RealizePalette hDCMem
    End If
    
    If hPal Then
        DeleteObject hPal
        hPal = 0
    End If
    
    ReleaseDC 0&, hDCMem
End If ' hDCMem

If result Then
    If BI.hBmp Then
        DeleteObject BI.hBmp
        BI.hBmp = 0
    End If
    
    ' Load the BitmapInfo structure
    
    With BI
        .hBmp = hMonoBmp
        .Width = bmih.biWidth
        .Height = Abs(bmih.biHeight)
        .XPelsPerMeter = bmih.biXPelsPerMeter
        .YPelsPerMeter = bmih.biYPelsPerMeter
    End With
    
    DIBtoMonoBitmap = BITMAP_OK
End If ' result

Exit Function

' - - - - - - - - Error Handler and Exit  - - - - - -
DMBErrs:

DIBtoMonoBitmap = BITMAP_ERRS

DMBExit:

If hMonoBmp Then
    DeleteObject hMonoBmp
    hMonoBmp = 0
End If
If hClrBmp Then
    DeleteObject hClrBmp
    hClrBmp = 0
End If
If hPalOld Then
    hPal = SelectPalette(hDCMem, hPalOld, False)
    hPalOld = 0
    RealizePalette hDCMem
End If
If hPal Then
    DeleteObject hPal
    hPal = 0
End If
If hDCMem Then ReleaseDC 0, hDCMem

Exit Function

End Function ' DIBtoMonoBitmap

'---------------------------------------------------------------------------
' Opens a bitmap file and creates a memory mono bitmap (relevant info is
' stored in the BitmapInfo structure).
' Assumes the input file exists.
'
Private Function GetBitmap(BI As BMPINFO, ByVal File As String) As Long

Dim bmfh                As BITMAPFILEHEADER
Dim FH                  As Long         ' file handle
Dim bmib()              As Byte         ' BITMAPINFO as a byte array
Dim Data()              As Byte         ' bitmap data

On Error GoTo GBErrs

GetBitmap = BITMAP_NOTBITMAP

' Warning "Open for binary" will try to create a file if it doesn't exist

FH = FreeFile
Open File For Binary Access Read Shared As #FH
Get #FH, , bmfh

' Validate BITMAPFILEHEADER

If bmfh.bfType <> BFT_BITMAP Then GoTo GBExit
If bmfh.bfSize <> LOF(FH) Then GoTo GBExit

' Extract the header and colour palette

ReDim bmib(0 To bmfh.bfOffBits - Len(bmfh) - 1)

Get #FH, Len(bmfh) + 1, bmib()

' Extract the data

ReDim Data(0 To bmfh.bfSize - bmfh.bfOffBits - 1)
Get #FH, bmfh.bfOffBits + 1, Data()
Close #FH
FH = 0

GetBitmap = DIBtoMonoBitmap(bmib(), Data(), BI)

GoTo GBExit

Exit Function

' - - - - - - - - Error Handler and Exit  - - - - - -
GBErrs:

GetBitmap = BITMAP_ERRS

GBExit:

If GetBitmap = BITMAP_ERRS Then MsgBox Err.Description, vbExclamation
If GetBitmap = BITMAP_FAIL Then MsgBox "Failed to load bitmap", vbExclamation
If GetBitmap = BITMAP_NOTBITMAP Then MsgBox "File is not a bitmap", vbExclamation

If FH Then Close #FH

Exit Function

End Function ' GetBitmap

'---------------------------------------------------------------------------
' Get a bitmap held in a global memory block DIB
'
Private Function GetMonoBitmapFromDIB(BI As BMPINFO, ByVal hDIB As Long, Optional ByVal AllowWarning As Boolean = True) As Boolean

Dim bmi                 As BITMAPINFO
Dim NumBytes            As Long         ' number of bytes required to hold the DIB
Dim lpDIB               As Long         ' pointer to memory
Dim bmib()              As Byte         ' BITMAPINFO as a byte array
Dim Data()              As Byte         ' bitmap data

GetMonoBitmapFromDIB = False

On Error GoTo GMBFDErrs

lpDIB = GlobalLock(hDIB)
If lpDIB Then

    ' Get the bitmap header
    
    CopyMemory bmi.bmiHeader, ByVal lpDIB, Len(bmi.bmiHeader)

    ' Calculate the size required for byte arrays
    
    ReDim bmi.bmiColors(0 To 0)
    NumBytes = bmi.bmiHeader.biSize + Len(bmi.bmiColors(0)) * DIBNumColours(bmi.bmiHeader, False)
    
    ReDim bmib(0 To NumBytes - 1)
    
    CopyMemory bmib(0), ByVal lpDIB, NumBytes
    lpDIB = lpDIB + NumBytes
    
    With bmi.bmiHeader
        NumBytes = Int((.biWidth * .biBitCount + 31) / 32) * 4 * Abs(.biHeight)
    End With
    
    ReDim Data(0 To NumBytes - 1)
    CopyMemory Data(0), ByVal lpDIB, NumBytes

    GlobalUnlock hDIB
    hDIB = 0

    If DIBtoMonoBitmap(bmib(), Data(), BI) = BITMAP_OK Then
        GetMonoBitmapFromDIB = True
    End If

   
End If ' lpDIB

Exit Function

' - - - - - - - - Error Handler - - - - - - - - - - -
GMBFDErrs:

If hDIB Then GlobalUnlock hDIB

If AllowWarning Then
    MsgBox "Failed to retrieve image" & vbCrLf & vbCrLf & Err.Description, vbCritical
End If

Exit Function

End Function ' GetMonoBitmapFromDIB

'---------------------------------------------------------------------------
' Retrieve the results from the service process and load into 'Results'
' Remember the character numbers returned refer to the Windows character set.
'
Private Function GetResults(ByVal JobNo As Long, Results As TOCRRESULTS) As Boolean

Dim ResultsInf          As Long         ' number of bytes needed for results
Dim Bytes()             As Byte         ' work array

GetResults = False
Results.Hdr.NumItems = 0

If TOCRGetJobResults(JobNo, ResultsInf, ByVal 0&) = TOCR_OK Then
    If ResultsInf > 0 Then
        ReDim Bytes(0 To ResultsInf - 1)
        If TOCRGetJobResults(JobNo, ResultsInf, Bytes(0)) = TOCR_OK Then
            UnpackResults Bytes(), Results
            With Results
                If .Hdr.StructId = 0 Then
                    If .Hdr.NumItems > 0 Then
                        If .Item(0).StructId <> 0 Then
                            MsgBox "Wrong results item structure Id" & _
                                Str$(.Item(0).StructId), vbCritical
                            .Hdr.NumItems = 0
                        End If
                    End If
                Else
                    MsgBox "Wrong results header structure Id" & _
                        Str$(.Hdr.StructId), vbCritical
                End If
            End With ' results
        End If
        GetResults = True
    End If
End If

End Function ' GetResults

'---------------------------------------------------------------------------
' Save a bitmap held in memory to a memory (only) mapped file.
' The handle to the MM file is returned in hFile.
'
Private Function SaveMonoBitmapToMMFile(BI As BMPINFO, hFile As Long) As Boolean

Const PAGE_READWRITE = 4
Const FILE_MAP_WRITE = 2

Dim bmi                 As BITMAPINFO
Dim ScanWidth           As Long         ' scan line width in bytes
Dim NumBytes            As Long         ' number of bytes required to hold the DIB
Dim lpMap               As Long         ' pointer to mapped file
Dim lp                  As Long         ' pointer
Dim hDCMem              As Long         ' handle to a memory device context
Dim hbmpOld             As Long         ' handle to a bitmap

SaveMonoBitmapToMMFile = False
hFile = 0

' Initialise the header

With bmi.bmiHeader
    .biSize = Len(bmi.bmiHeader)
    .biWidth = BI.Width
    .biHeight = BI.Height
    .biXPelsPerMeter = BI.XPelsPerMeter
    .biYPelsPerMeter = BI.YPelsPerMeter
    .biPlanes = 1
    .biBitCount = 1
    .biCompression = BI_RGB
    .biClrUsed = 2
    .biClrImportant = 0
    ScanWidth = Int((BI.Width + 31) / 32) * 4
    .biSizeImage = ScanWidth * BI.Height
End With

' Initialise the palette

ReDim bmi.bmiColors(0 To 1)

With bmi.bmiColors(0)
    .rgbRed = 0
    .rgbGreen = 0
    .rgbBlue = 0
    .rgbReserved = 0
End With
With bmi.bmiColors(1)
    .rgbRed = 255
    .rgbGreen = 255
    .rgbBlue = 255
    .rgbReserved = 0
End With

' Calculate the size of the memory mapped file memory

NumBytes = Len(bmi.bmiHeader) + Len(bmi.bmiColors(0)) * 2 + bmi.bmiHeader.biSizeImage

' Create a memory only file

hFile = CreateFileMappingMy(&HFFFFFFFF, ByVal 0&, PAGE_READWRITE, 0, NumBytes, ByVal 0&)
If hFile Then
    lpMap = MapViewOfFile(hFile, FILE_MAP_WRITE, 0, 0, 0)
    If lpMap Then
    
        ' Copy the bitmap header to the MM file
        
        lp = lpMap
        CopyMemory ByVal lp, bmi.bmiHeader, Len(bmi.bmiHeader)
        lp = lp + Len(bmi.bmiHeader)
        CopyMemory ByVal lp, bmi.bmiColors(0), Len(bmi.bmiColors(0)) * 2
        lp = lp + Len(bmi.bmiColors(0)) * 2
        
        ' Retrieve the bitmap bits and copy to the MM file
        
        hDCMem = CreateCompatibleDC(0)
        If hDCMem Then
            If GetDIBitsMy(hDCMem, BI.hBmp, 0, BI.Height, ByVal lp, ByVal lpMap, DIB_RGB_COLORS) Then
                SaveMonoBitmapToMMFile = True
            End If
            DeleteDC hDCMem
        End If ' hDCMem
        
        UnmapViewOfFile ByVal lpMap
        
    End If ' lpMap
End If ' hFile

If Not SaveMonoBitmapToMMFile Then
    If hFile Then
        CloseHandle hFile
        hFile = 0
    End If
End If

End Function ' SaveMonoBitmapToMMFile

'---------------------------------------------------------------------------
' This routine provides one solution to the classic VB problem of how do you
' read 'variable structures' (structures which contain a variable number of
' another structure) in VB.  There are many examples of this in the Windows
' API (BITMAPINFO being one). This typically occurs when chatting to DLLs
' written in/for C because C, having no array bound checking, has no
' difficulty with the problem.
'
' This routine assumes the required data has been read into the Byte array
' 'Bytes' and then it re-dimensions 'Results' appropriately and copies the
' data.
Private Sub UnpackResults(Bytes() As Byte, Results As TOCRRESULTS)

Dim HeaderLen           As Long         ' length of TOCRRESULTSHEADER
Dim ItemLen             As Long         ' length of TOCRRESULTSITEM
Dim ResultsLen          As Long         ' real length of Results
Dim NumItems            As Long         ' number of items in Results

' Notice the use of LenB here (see VB help for difference between Len and LenB)

HeaderLen = LenB(Results.Hdr)
ItemLen = LenB(Results.Item(0))

' Find the number of items in the array

ResultsLen = UBound(Bytes) - LBound(Bytes) + 1
NumItems = (ResultsLen - HeaderLen) / ItemLen

' Copy the header

CopyMemory Results, Bytes(LBound(Bytes)), HeaderLen


' Copy the array of items

If NumItems > 0 Then
    ReDim Results.Item(0 To NumItems - 1)
    CopyMemory Results.Item(0), Bytes(LBound(Bytes) + HeaderLen), ItemLen * NumItems
End If

' Note, the reason you need two CopyMemorys is because 'Item()' in 'Results'
' is in fact just a pointer to the array of items.  You can verify this by
' find LenB(Results), re-dimension Results.Items(), refind LenB(Results) - it
' will be unchanged.
'
' Had 'Items()' been a fixed array in Results (dimensioned to some value in the
' Type declaration) then this routine will still work but you wouldn't need it
' because you could have just sent 'Results' to the API call.

End Sub ' UnpackResults

Private Function FormatResults(Results As TOCRRESULTS, Answer As String) As Boolean

Dim ItemNo As Integer

FormatResults = False
Answer = ""

With Results
    If .Hdr.NumItems > 0 Then
        For ItemNo = 0 To .Hdr.NumItems - 1
            If Chr(.Item(ItemNo).OCRCha) = vbCr Then
                Answer = Answer & vbCrLf
            Else
                Answer = Answer & Chr(.Item(ItemNo).OCRCha)
            End If
        Next ItemNo
        FormatResults = True
    Else
        MsgBox "No results returned", vbInformation, "FormatResults"
    End If
End With

End Function ' Format Results


